<?php
declare(strict_types=1);

// Prevent direct access
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    exit('Method Not Allowed');
}

// Load dependencies
require_once __DIR__ . '/../includes/db.php';

// PHPMailer classes (User must ensure these files exist in ../libs/PHPMailer/src/)
// If using Composer, require 'vendor/autoload.php' instead.
if (file_exists(__DIR__ . '/../libs/PHPMailer/src/PHPMailer.php')) {
    require_once __DIR__ . '/../libs/PHPMailer/src/Exception.php';
    require_once __DIR__ . '/../libs/PHPMailer/src/PHPMailer.php';
    require_once __DIR__ . '/../libs/PHPMailer/src/SMTP.php';
}
else {
    // Fallback or error logging if library is missing
    error_log("PHPMailer library missing in ../libs/PHPMailer/src/");
    http_response_code(500);
    exit('Email System Error');
}

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\SMTP;
use PHPMailer\PHPMailer\Exception;

try {
    // 1. Fetch System Settings (SMTP & Webhook Secret)
    $stmt = $pdo->query("SELECT * FROM settings WHERE id = 1 LIMIT 1");
    $settings = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$settings || empty($settings['lemon_secret'])) {
        error_log("Webhook Error: Missing Lemon Squeezy Secret in Settings.");
        http_response_code(500);
        exit('Server Configuration Error');
    }

    $lemon_secret = $settings['lemon_secret'];

    // 2. Read Raw Payload & Signature
    $payload = file_get_contents('php://input');
    $signature_header = $_SERVER['HTTP_X_SIGNATURE'] ?? '';

    // 3. Cryptographic Verification
    $hash = hash_hmac('sha256', $payload, $lemon_secret);

    if (!hash_equals($hash, $signature_header)) {
        http_response_code(401);
        exit('Unauthorized: Invalid Signature');
    }

    // 4. Process Payload
    $data = json_decode($payload, true);

    // Check if event is 'order_created' or similar
    $event_name = $data['meta']['event_name'] ?? '';

    if ($event_name !== 'order_created') {
        // Just return 200 for other events to acknowledge receipt
        http_response_code(200);
        exit('Event Ignored');
    }

    // Extract Buyer Email
    $client_email = $data['data']['attributes']['user_email'] ?? '';

    if (empty($client_email)) {
        error_log("Webhook Error: No email found in payload.");
        http_response_code(400);
        exit('Invalid Payload');
    }

    // 5. Generate License Key
    $new_key = 'BIO-' . strtoupper(bin2hex(random_bytes(8)));

    // 6. Insert into Database
    // Note: 'licenses' table structure from includes/db.php:
    // license_key, client_email, tier, max_domains, registered_domains, status
    $stmt = $pdo->prepare("INSERT INTO licenses (license_key, client_email, status, created_at) VALUES (:key, :email, 'active', CURRENT_TIMESTAMP)");
    $stmt->execute([
        ':key' => $new_key,
        ':email' => $client_email
    ]);

    // 7. Send Email via PHPMailer
    $mail = new PHPMailer(true);

    try {
        // Server settings
        // $mail->SMTPDebug = SMTP::DEBUG_OFF; // Disable verbose debug output
        $mail->isSMTP();
        $mail->Host = $settings['smtp_host'];
        $mail->SMTPAuth = true;
        $mail->Username = $settings['smtp_user'];
        $mail->Password = $settings['smtp_pass'];

        // Determine encryption based on port
        if ((int)$settings['smtp_port'] === 465) {
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
        }
        elseif ((int)$settings['smtp_port'] === 587) {
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
        }
        else {
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS; // Default
        }

        $mail->Port = (int)$settings['smtp_port'];

        // Recipients
        $mail->setFrom($settings['smtp_from_email'], $settings['smtp_from_name']);
        $mail->addAddress($client_email);

        // Content
        $mail->isHTML(true);
        $mail->Subject = 'Your BioScript License Key';

        // Email Template
        $mail->Body = "
        <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #e2e8f0; border-radius: 10px; background-color: #f8fafc;'>
            <div style='text-align: center; margin-bottom: 20px;'>
                <h1 style='color: #0f172a;'>Welcome to BioScript</h1>
            </div>
            <div style='background-color: #ffffff; padding: 20px; border-radius: 8px; box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);'>
                <p style='color: #334155; font-size: 16px;'>Thank you for your purchase!</p>
                <p style='color: #334155; font-size: 16px;'>Here is your official license key to activate your BioScript installation:</p>
                
                <div style='background-color: #f1f5f9; padding: 15px; border-radius: 6px; text-align: center; margin: 20px 0; border: 1px dashed #cbd5e1;'>
                    <code style='font-size: 20px; font-weight: bold; color: #2563eb; letter-spacing: 1px;'>{$new_key}</code>
                </div>
                
                <p style='color: #334155; font-size: 14px;'>You can use this key to activate your domain immediately.</p>
                
                <hr style='border: 0; border-top: 1px solid #e2e8f0; margin: 20px 0;'>
                
                <p style='color: #64748b; font-size: 12px; text-align: center;'>
                    If you have any questions, please reply to this email.
                </p>
            </div>
        </div>
        ";

        $mail->AltBody = "Thank you for your purchase! Your License Key is: {$new_key}";

        $mail->send();

    }
    catch (Exception $e) {
        error_log("Message could not be sent. Mailer Error: {$mail->ErrorInfo}");
    // We still return 200 because the license was generated successfully
    }

    http_response_code(200);
    echo "License Generated & Email Sent";

}
catch (PDOException $e) {
    error_log("Database Error: " . $e->getMessage());
    http_response_code(500);
    exit('Internal Server Error');
}
catch (Exception $e) {
    error_log("General Error: " . $e->getMessage());
    http_response_code(500);
    exit('Internal Server Error');
}