<?php
if (session_status() !== PHP_SESSION_ACTIVE) {
    session_start();
}
require_once __DIR__ . '/includes/db.php';

// --- AUTHENTICATION & CSRF SETUP ---

// Check Login
if (!isset($_SESSION['ls_admin_logged_in'])) {
    header('Location: index.php');
    exit;
}

// Generate CSRF Token
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

function verify_csrf() {
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        die('CSRF Token Validation Failed');
    }
}

// --- ACTIONS ---

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verify_csrf();
    
    if (isset($_POST['action']) && isset($_POST['order_id'])) {
        $order_id = $_POST['order_id'];
        $action = $_POST['action'];
        
        // Validate order exists
        $stmt = $pdo->prepare("SELECT * FROM orders WHERE woo_order_id = ?");
        $stmt->execute([$order_id]);
        $order = $stmt->fetch();
        
        if ($order && $order['status'] === 'completed') {
            if ($action === 'revoke') {
                $license = $order['license_key'];
                
                $pdo->beginTransaction();
                
                // Update Order Status
                $stmt = $pdo->prepare("UPDATE orders SET status = 'refunded', updated_at = CURRENT_TIMESTAMP WHERE woo_order_id = ?");
                $stmt->execute([$order_id]);
                
                // Revoke License if exists
                if ($license) {
                    $stmt = $pdo->prepare("UPDATE licenses SET status = 'revoked' WHERE license_key = ?");
                    $stmt->execute([$license]);
                }
                
                $pdo->commit();
                $success = "Order $order_id refunded and license revoked.";
            }
            
            if ($action === 'resend') {
                $email = $order['customer_email'];
                $license = $order['license_key'];
                
                if ($license && file_exists(__DIR__ . '/../libs/PHPMailer/src/PHPMailer.php')) {
                    require_once __DIR__ . '/../libs/PHPMailer/src/Exception.php';
                    require_once __DIR__ . '/../libs/PHPMailer/src/PHPMailer.php';
                    require_once __DIR__ . '/../libs/PHPMailer/src/SMTP.php';
                    
                    $stmtSettings = $pdo->query("SELECT * FROM settings WHERE id = 1 LIMIT 1");
                    $settings = $stmtSettings->fetch(PDO::FETCH_ASSOC);
                    
                    if ($settings) {
                        try {
                            $mail = new \PHPMailer\PHPMailer\PHPMailer(true);
                            $mail->isSMTP();
                            $mail->Host = $settings['smtp_host'];
                            $mail->SMTPAuth = true;
                            $mail->Username = $settings['smtp_user'];
                            $mail->Password = $settings['smtp_pass'];
                            $mail->SMTPSecure = (int)$settings['smtp_port'] === 465 ? \PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_SMTPS : \PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
                            $mail->Port = (int)$settings['smtp_port'];
                            $mail->setFrom($settings['smtp_from_email'], $settings['smtp_from_name']);
                            $mail->addAddress($email);
                            $mail->isHTML(true);
                            $mail->Subject = 'Your BioScript Architecture License Key (Resent)';
                            $mail->Body = "Your generated license key is: <strong>{$license}</strong>";
                            $mail->send();
                            $success = "License resent to $email successfully.";
                        } catch (Exception $e) {
                            $error = "Mail failed to send: " . $e->getMessage();
                        }
                    } else {
                        $error = "SMTP settings not configured.";
                    }
                } else {
                    $error = "Could not resend. Mailer library missing or no license generated.";
                }
            }
        }
    }
}

// --- PAGINATION & FILTERING ---

$filter = $_GET['filter'] ?? 'all';
$page = max(1, (int)($_GET['page'] ?? 1));
$limit = 50;
$offset = ($page - 1) * $limit;

$where = "1=1";
$params = [];
$valid_filters = ['completed', 'pending', 'failed', 'refunded'];
if (in_array(strtolower($filter), $valid_filters)) {
    $where .= " AND status = :status";
    $params[':status'] = strtolower($filter);
}

// Count total (No N+1 queries)
$stmt = $pdo->prepare("SELECT COUNT(*) FROM orders WHERE $where");
$stmt->execute($params);
$total_orders = (int)$stmt->fetchColumn();
$total_pages = max(1, ceil($total_orders / $limit));

// Fetch rows with Pagination
$stmt = $pdo->prepare("SELECT * FROM orders WHERE $where ORDER BY created_at DESC LIMIT :limit OFFSET :offset");
foreach ($params as $k => $v) {
    $stmt->bindValue($k, $v);
}
$stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

?>
<!DOCTYPE html>
<html lang="en" class="dark">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>WooCommerce Orders - License Authority</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            darkMode: 'class',
            theme: {
                extend: {
                    colors: {
                        slate: { 850: '#1e293b', 950: '#0f172a' },
                        primary: { 500: '#6366f1', 600: '#4f46e5' }
                    },
                    fontFamily: {
                        sans: ['Inter', 'sans-serif'],
                        mono: ['JetBrains Mono', 'monospace'],
                    }
                }
            }
        }
    </script>
    <link
        href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&family=JetBrains+Mono:wght@400;500&display=swap"
        rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3.x.x/dist/cdn.min.js"></script>
    <style>
        .ui-enterprise {
            font-family: 'Inter', sans-serif;
            --ent-primary: #2563eb;
            --ent-primary-dark: #1d4ed8;
            --ent-surface: #0f172a;
            --ent-card: #1e293b;
            --ent-border: #334155;
            --ent-border-strong: #475569;
            --ent-text: #f8fafc;
            --ent-muted: #94a3b8;
            --ent-accent: #3b82f6;
            --ent-success: #10b981;
            --ent-warning: #f59e0b;
            --ent-danger: #ef4444;
        }

        .ui-enterprise .ent-glass {
            background: rgba(15, 23, 42, 0.85);
            backdrop-filter: blur(20px);
            -webkit-backdrop-filter: blur(20px);
            border: 1px solid var(--ent-border-strong);
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.5);
        }

        .ui-enterprise .ent-btn-primary {
            background: var(--ent-primary);
            color: white;
            transition: all 0.2s;
            box-shadow: 0 2px 4px rgba(37, 99, 235, 0.3);
        }

        .ui-enterprise .ent-btn-primary:hover {
            background: var(--ent-primary-dark);
            box-shadow: 0 4px 12px rgba(37, 99, 235, 0.4);
        }

        .ui-enterprise .ent-btn-primary:active {
            transform: scale(0.98);
        }

        /* Custom Scrollbar */
        .ui-enterprise ::-webkit-scrollbar {
            width: 6px;
            height: 6px;
        }

        .ui-enterprise ::-webkit-scrollbar-track {
            background: var(--ent-surface);
        }

        .ui-enterprise ::-webkit-scrollbar-thumb {
            background: var(--ent-border-strong);
            border-radius: 3px;
        }

        .ui-enterprise ::-webkit-scrollbar-thumb:hover {
            background: var(--ent-muted);
        }

        /* Nav Pills for Filters */
        .ent-pill {
            padding: 0.5rem 1rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 0.05em;
            transition: all 0.2s;
            border: 1px solid transparent;
        }

        .ent-pill-active {
            background: rgba(59, 130, 246, 0.1);
            color: #60a5fa;
            border-color: rgba(59, 130, 246, 0.3);
        }

        .ent-pill-inactive {
            color: #94a3b8;
            border-color: #334155;
        }

        .ent-pill-inactive:hover {
            background: #1e293b;
            color: #f8fafc;
        }
    </style>
</head>

<body class="ui-enterprise bg-slate-950 text-slate-100 h-screen flex overflow-hidden font-sans">

    <!-- Sidebar (Existing Layout) -->
    <aside class="w-64 bg-slate-950 border-r border-slate-800 flex flex-col hidden md:flex shrink-0 z-20">
        <div class="p-6 border-b border-slate-800 flex items-center space-x-3">
            <div class="w-8 h-8 bg-blue-600 rounded flex items-center justify-center border border-blue-500 shadow-sm">
                <i class="fas fa-layer-group text-white text-sm"></i>
            </div>
            <div>
                <h1 class="text-base font-black text-white tracking-widest uppercase leading-tight">Guard<span
                        class="text-blue-500">Node</span></h1>
                <p class="text-[9px] text-slate-500 font-bold uppercase tracking-widest">Digital Infrastructure</p>
            </div>
        </div>

        <nav class="flex-1 py-6 space-y-1">
            <p class="px-6 text-[10px] font-black text-slate-500 uppercase tracking-widest mb-3">Control Plane</p>

            <a href="dashboard.php"
                class="flex items-center space-x-3 px-6 py-3 text-slate-400 hover:bg-slate-900 border-l-2 border-transparent hover:border-slate-700 transition-all group">
                <i class="fas fa-satellite-dish w-5 text-center group-hover:text-slate-300 transition-colors"></i>
                <span class="font-semibold tracking-wide text-sm">License Overview</span>
            </a>

            <a href="orders.php"
                class="flex items-center space-x-3 px-6 py-3 bg-slate-900 text-blue-400 border-l-2 border-blue-500 transition-all group">
                <i class="fas fa-shopping-cart w-5 text-center"></i>
                <span class="font-semibold tracking-wide text-sm">WooCommerce Orders</span>
            </a>

            <a href="settings.php"
                class="flex items-center space-x-3 px-6 py-3 text-slate-400 hover:bg-slate-900 border-l-2 border-transparent hover:border-slate-700 transition-all group">
                <i class="fas fa-sliders-h w-5 text-center group-hover:text-slate-300 transition-colors"></i>
                <span class="font-semibold tracking-wide text-sm">Global Settings</span>
            </a>
        </nav>

        <div class="p-4 border-t border-slate-800">
            <a href="index.php?logout=true"
                class="flex items-center justify-center space-x-2 w-full px-4 py-2 hover:bg-slate-900 text-slate-500 hover:text-red-400 border border-transparent hover:border-slate-800 rounded transition-all text-xs font-bold uppercase tracking-wider">
                <i class="fas fa-power-off"></i>
                <span>Terminate Session</span>
            </a>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="flex-1 overflow-y-auto p-8 lg:p-12 relative">
        <!-- Background Glow -->
        <div
            class="absolute top-0 left-0 w-full h-96 bg-gradient-to-b from-primary-900/20 to-transparent pointer-events-none">
        </div>

        <!-- Header -->
        <header class="flex justify-between items-end mb-10 relative z-10">
            <div>
                <h2 class="text-3xl font-bold text-white mb-2 tracking-tight">Order Logs</h2>
                <p class="text-slate-400">Track and manage inbound WooCommerce webhook executions.</p>
            </div>
            <div class="flex space-x-2">
                <a href="orders.php?filter=all"
                    class="ent-pill <?php echo $filter === 'all' ? 'ent-pill-active' : 'ent-pill-inactive'; ?>">All</a>
                <a href="orders.php?filter=completed"
                    class="ent-pill <?php echo $filter === 'completed' ? 'ent-pill-active' : 'ent-pill-inactive'; ?>">Completed</a>
                <a href="orders.php?filter=pending"
                    class="ent-pill <?php echo $filter === 'pending' ? 'ent-pill-active' : 'ent-pill-inactive'; ?>">Pending</a>
                <a href="orders.php?filter=failed"
                    class="ent-pill <?php echo $filter === 'failed' ? 'ent-pill-active' : 'ent-pill-inactive'; ?>">Failed</a>
                <a href="orders.php?filter=refunded"
                    class="ent-pill <?php echo $filter === 'refunded' ? 'ent-pill-active' : 'ent-pill-inactive'; ?>">Refunded</a>
            </div>
        </header>

        <?php if (isset($success)): ?>
        <div
            class="bg-emerald-500/10 border border-emerald-500/20 text-emerald-400 p-4 rounded-xl mb-8 flex items-center relative z-10 animate-fade-in-down">
            <i class="fas fa-check-circle mr-3 text-lg"></i>
            <span>
                <?php echo htmlspecialchars($success); ?>
            </span>
        </div>
        <?php endif; ?>

        <?php if (isset($error)): ?>
        <div
            class="bg-red-500/10 border border-red-500/20 text-red-400 p-4 rounded-xl mb-8 flex items-center relative z-10 animate-fade-in-down">
            <i class="fas fa-exclamation-circle mr-3 text-lg"></i>
            <span>
                <?php echo htmlspecialchars($error); ?>
            </span>
        </div>
        <?php endif; ?>

        <!-- Orders Table -->
        <div class="ent-glass rounded-2xl shadow-xl overflow-hidden relative z-10">
            <div class="overflow-x-auto">
                <table class="w-full text-left text-sm text-slate-400">
                    <thead
                        class="bg-slate-900/80 text-slate-300 uppercase text-xs font-bold tracking-wider border-b border-slate-700/50">
                        <tr>
                            <th class="px-6 py-5">Order ID</th>
                            <th class="px-6 py-5">Email</th>
                            <th class="px-6 py-5">Amount</th>
                            <th class="px-6 py-5">Status</th>
                            <th class="px-6 py-5">License</th>
                            <th class="px-6 py-5">Date</th>
                            <th class="px-6 py-5 text-right">Actions</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-slate-800/50">
                        <?php foreach ($orders as $order): 
                            
                            $status_class = match(strtolower($order['status'])) {
                                'completed' => 'bg-emerald-500/10 text-emerald-400 border-emerald-500/20',
                                'pending' => 'bg-amber-500/10 text-amber-500 border-amber-500/20',
                                'failed' => 'bg-red-500/10 text-red-400 border-red-500/20',
                                'refunded' => 'bg-purple-500/10 text-purple-400 border-purple-500/20',
                                default => 'bg-slate-800 text-slate-300 border-slate-700'
                            };

                            $date = date('M d, Y H:i', strtotime($order['created_at']));
                        ?>
                        <tr class="hover:bg-slate-800/40 transition-colors group">
                            <td class="px-6 py-4 font-mono text-white text-xs">
                                #
                                <?php echo htmlspecialchars($order['woo_order_id']); ?>
                            </td>
                            <td class="px-6 py-4 text-white font-medium">
                                <?php echo htmlspecialchars($order['customer_email']); ?>
                            </td>
                            <td class="px-6 py-4 font-mono text-slate-300">
                                <?php echo htmlspecialchars($order['amount'] . ' ' . $order['currency']); ?>
                            </td>
                            <td class="px-6 py-4">
                                <span
                                    class="inline-flex items-center px-2.5 py-1 rounded text-[10px] uppercase font-bold border <?php echo $status_class; ?>">
                                    <?php echo htmlspecialchars($order['status']); ?>
                                </span>
                            </td>
                            <td class="px-6 py-4">
                                <?php if ($order['license_key']): ?>
                                <div class="font-mono text-xs text-blue-400">
                                    <?php echo htmlspecialchars($order['license_key']); ?>
                                </div>
                                <?php else: ?>
                                <span
                                    class="text-slate-600 text-[10px] uppercase tracking-widest font-bold">Unassigned</span>
                                <?php endif; ?>
                            </td>
                            <td class="px-6 py-4 text-xs text-slate-500">
                                <?php echo $date; ?>
                            </td>
                            <td class="px-6 py-4 text-right">
                                <?php if ($order['status'] === 'completed'): ?>
                                <div class="flex items-center justify-end space-x-2">

                                    <form method="POST" class="inline">
                                        <input type="hidden" name="csrf_token"
                                            value="<?php echo $_SESSION['csrf_token']; ?>">
                                        <input type="hidden" name="action" value="resend">
                                        <input type="hidden" name="order_id"
                                            value="<?php echo htmlspecialchars($order['woo_order_id']); ?>">
                                        <button type="submit"
                                            class="bg-blue-500/10 hover:bg-blue-500/20 text-blue-400 border border-blue-500/20 rounded px-2 py-1 text-[10px] font-bold uppercase tracking-wider transition-colors"
                                            title="Resend Email">
                                            Resend
                                        </button>
                                    </form>

                                    <form method="POST" class="inline"
                                        onsubmit="return confirm('Are you sure you want to refund this order and permanently revoke the generated license?');">
                                        <input type="hidden" name="csrf_token"
                                            value="<?php echo $_SESSION['csrf_token']; ?>">
                                        <input type="hidden" name="action" value="revoke">
                                        <input type="hidden" name="order_id"
                                            value="<?php echo htmlspecialchars($order['woo_order_id']); ?>">
                                        <button type="submit"
                                            class="bg-red-500/10 hover:bg-red-500/20 text-red-400 border border-red-500/20 rounded px-2 py-1 text-[10px] font-bold uppercase tracking-wider transition-colors"
                                            title="Revoke License">
                                            Revoke
                                        </button>
                                    </form>

                                </div>
                                <?php elseif ($order['status'] === 'refunded'): ?>
                                <span
                                    class="inline-flex items-center text-[10px] font-bold uppercase tracking-widest text-purple-500">
                                    <i class="fas fa-ban mr-1"></i> Revoked
                                </span>
                                <?php else: ?>
                                <span class="text-slate-600 text-[10px] font-bold uppercase tracking-widest">—</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                <?php if (count($orders) === 0): ?>
                <div class="p-16 text-center text-slate-500">
                    <i class="fas fa-receipt text-3xl mb-4 opacity-30"></i>
                    <p class="text-xs font-bold uppercase tracking-widest">No orders found matching criteria</p>
                </div>
                <?php endif; ?>
            </div>

            <!-- Pagination Controls -->
            <?php if ($total_pages > 1): ?>
            <div class="px-6 py-4 bg-slate-900/50 border-t border-slate-700/50 flex justify-between items-center">
                <span class="text-xs font-bold text-slate-500 uppercase tracking-widest">Page
                    <?php echo $page; ?> of
                    <?php echo $total_pages; ?>
                </span>
                <div class="flex space-x-2">
                    <?php if ($page > 1): ?>
                    <a href="?filter=<?php echo urlencode($filter); ?>&page=<?php echo $page - 1; ?>"
                        class="px-3 py-1 bg-slate-800 text-white rounded hover:bg-slate-700 text-xs font-bold transition-colors">Prev</a>
                    <?php endif; ?>

                    <?php if ($page < $total_pages): ?>
                    <a href="?filter=<?php echo urlencode($filter); ?>&page=<?php echo $page + 1; ?>"
                        class="px-3 py-1 bg-slate-800 text-white rounded hover:bg-slate-700 text-xs font-bold transition-colors">Next</a>
                    <?php endif; ?>
                </div>
            </div>
            <?php endif; ?>

        </div>
    </main>
</body>

</html>